/*
 * Decompiled with CFR 0.152.
 */
package com.enhancedplayerlist.server;

import com.enhancedplayerlist.Config;
import com.enhancedplayerlist.data.PlayerStatsData;
import com.enhancedplayerlist.network.NetworkHandler;
import com.enhancedplayerlist.network.PlayerStatsPacket;
import com.google.gson.Gson;
import com.google.gson.JsonObject;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.Reader;
import java.time.Duration;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import net.minecraft.server.MinecraftServer;
import net.minecraft.server.level.ServerPlayer;
import net.minecraft.world.entity.player.Player;
import net.minecraft.world.level.storage.LevelResource;

public class ServerStatsManager {
    private static final Gson GSON = new Gson();
    private static final Map<UUID, PlayerStatsData> playerStats = new ConcurrentHashMap<UUID, PlayerStatsData>();
    private static final Map<UUID, Long> fileModificationCache = new ConcurrentHashMap<UUID, Long>();
    private static final Duration CLEANUP_THRESHOLD = Duration.ofDays(30L);
    private static final long FILE_CACHE_DURATION = 5000L;
    private static MinecraftServer server;

    public static void init(MinecraftServer server) {
        ServerStatsManager.server = server;
        ServerStatsManager.loadAllPlayerStats();
        ServerStatsManager.cleanupOldPlayers();
    }

    private static void cleanupOldPlayers() {
        if (!Config.showOfflinePlayers) {
            long now = System.currentTimeMillis();
            playerStats.values().removeIf(stats -> !stats.isOnline() && now - stats.getLastSeen() > CLEANUP_THRESHOLD.toMillis());
        }
    }

    public static void loadAllPlayerStats() {
        if (server == null) {
            return;
        }
        File statsDir = server.getWorldPath(LevelResource.PLAYER_STATS_DIR).toFile();
        if (!statsDir.exists() || !statsDir.isDirectory()) {
            return;
        }
        File[] statFiles = statsDir.listFiles((dir, name) -> name.endsWith(".json"));
        if (statFiles == null) {
            return;
        }
        boolean dataUpdated = false;
        long now = System.currentTimeMillis();
        HashSet<UUID> processedUuids = new HashSet<UUID>();
        for (File statFile : statFiles) {
            try {
                String uuidStr = statFile.getName().replace(".json", "");
                UUID uuid2 = UUID.fromString(uuidStr);
                processedUuids.add(uuid2);
                long lastModified = statFile.lastModified();
                Long cachedModified = fileModificationCache.get(uuid2);
                if (cachedModified != null && cachedModified == lastModified) continue;
                fileModificationCache.put(uuid2, lastModified);
                JsonObject jsonStats = (JsonObject)GSON.fromJson((Reader)new BufferedReader(new FileReader(statFile)), JsonObject.class);
                PlayerStatsData oldData = playerStats.get(uuid2);
                PlayerStatsData.Builder newDataBuilder = PlayerStatsData.builder().uuid(uuidStr).lastSeen(lastModified);
                if (oldData != null) {
                    newDataBuilder.online(oldData.isOnline()).playerName(oldData.getPlayerName());
                    if (!oldData.isOnline()) {
                        newDataBuilder.lastSeen(lastModified);
                    }
                }
                PlayerStatsData newData = newDataBuilder.build();
                newData.loadFromJson(jsonStats);
                ServerPlayer player = server.getPlayerList().getPlayer(uuid2);
                if (player != null) {
                    newData.setOnline(true);
                    newData.setPlayerName(player.getGameProfile().getName());
                    newData.setLastSeen(System.currentTimeMillis());
                } else if (newData.getPlayerName().isEmpty()) {
                    Optional.ofNullable(server.getProfileCache()).flatMap(cache -> cache.get(uuid2)).ifPresent(profile -> newData.setPlayerName(profile.getName()));
                }
                playerStats.put(uuid2, newData);
                dataUpdated = true;
            }
            catch (Exception e) {
                e.printStackTrace();
            }
        }
        fileModificationCache.entrySet().removeIf(entry -> now - (Long)entry.getValue() > 5000L);
        if (!Config.showOfflinePlayers) {
            playerStats.keySet().removeIf(uuid -> !processedUuids.contains(uuid));
        }
        if (dataUpdated) {
            ServerStatsManager.syncToClients();
        }
    }

    public static void syncToClients() {
        if (server == null) {
            return;
        }
        Map visibleStats = playerStats.entrySet().stream().filter(e -> ((PlayerStatsData)e.getValue()).isOnline() || Config.showOfflinePlayers).collect(HashMap::new, (m, e) -> m.put((UUID)e.getKey(), (PlayerStatsData)e.getValue()), HashMap::putAll);
        if (!visibleStats.isEmpty()) {
            PlayerStatsPacket packet = new PlayerStatsPacket(visibleStats);
            NetworkHandler.sendToAllPlayers(packet);
        }
    }

    public static void forceSync() {
        if (server != null) {
            ServerStatsManager.loadAllPlayerStats();
            ServerStatsManager.syncToClients();
        }
    }

    public static void onPlayerJoin(Player player) {
        UUID uuid = player.getUUID();
        PlayerStatsData data = playerStats.computeIfAbsent(uuid, k -> PlayerStatsData.builder().uuid(k.toString()).playerName(player.getGameProfile().getName()).online(true).lastSeen(System.currentTimeMillis()).build());
        if (data != null) {
            data.setOnline(true);
            data.setPlayerName(player.getGameProfile().getName());
            ServerStatsManager.syncToClients();
        }
    }

    public static void onPlayerLeave(Player player) {
        UUID uuid = player.getUUID();
        PlayerStatsData data = playerStats.get(uuid);
        if (data != null) {
            data.setOnline(false);
            ServerStatsManager.syncToClients();
        }
    }

    public static void onServerStopping() {
        server = null;
        playerStats.clear();
        fileModificationCache.clear();
    }
}

